'use strict';
(function ($) {
    class GameService {
        getGames(args) {
            const defaultParams = {
                'page': 1,
                'per_page': 40,
            };
            const params = {
                ...defaultParams,
                ...args,
            };

            const action = 'fdgames_get_games';
            return new Promise((resolve, reject) => {
                $.ajax({
                    'url': fdgames_params?.ajax_url || '/wp-admin/admin-ajax.php',
                    'method': 'GET',
                    'dataType': 'json',
                    'data': {
                        action: action,
                        ...params,
                    },
                    'success': function (response) {
                        if (response.success === true && response?.data) {
                            resolve(response.data);
                        } else {
                            reject(response?.msg || 'Invalid response from server');
                        }
                    },
                    'error': function (xhr, status, error) {
                        reject(error || xhr.responseText || 'Unknown error');
                    },
                    'complete': function () {
                    }
                });
            });
        }
    }

    class RenderService {
        renderGames($grid, htmlArray, replace = true) {
            if (replace) $grid.empty();
            if (Array.isArray(htmlArray)) {
                htmlArray.forEach((html) => $grid.append(html));
            }
        }
    }

    class BrowserUrlService {
        qs;

        constructor() {
            this.qs = new URLSearchParams(location.search);
        }

        setQS(key, val) {
            if (val === undefined || val === null || String(val).trim() === '') this.qs.delete(key); else this.qs.set(key, val);
        };

        refreshFromLocation() {
            this.qs = new URLSearchParams(location.search);
        }

        updateURL(push = false) {
            const url = `${location.pathname}?${this.qs.toString()}`;
            if (push) {
                history.pushState(null, '', url);
            } else {
                history.replaceState(null, '', url);
            }
        };

        getQS(key, defaultVal = null) {
            return this.qs.get(key) ?? defaultVal;
        }
    }

    const renderService = new RenderService();
    const gameService = new GameService();
    const browserUrlService = new BrowserUrlService();

    $(document).ready(function () {
        filterFormInit();
        loadMoreInit();
    });

    function loadMoreInit() {
        $('.fdgames-load-more-btn').on('click', async function (e) {
            e.preventDefault();
            const loadMoreButton = $(this);
            const gamesContainer = loadMoreButton.closest('.fgames-cards-wrapper');
            const gamesGrid = gamesContainer.find('.fdgames-grid');
            const loadMoreContainer = loadMoreButton.closest('.fdgames-load-more-container');
            const totalMeta = loadMoreContainer.find('.fdgames-inthelist');

            loadMoreButton.attr('disabled', true);

            const viewed = parseInt(totalMeta.attr('data-viewed-games'), 10) || 0;
            const perPage = parseInt(totalMeta.attr('data-per-page'), 10) || 40;
            const totalAll = parseInt(totalMeta.attr('data-total'), 10) || null;

            const currentPage = Math.max(1, Math.ceil(viewed / perPage));
            const nextPage = currentPage + 1;

            // get filters from the URL
            const nameVal = browserUrlService.getQS('fdg-name', '') || '';
            const providerVal = browserUrlService.getQS('fdg-provider', '') || '';
            const sortVal = browserUrlService.getQS('fdg-sort', '') || '';

            try {
                const data = await gameService.getGames({
                    'page': nextPage,
                    'per_page': perPage,
                    'fdg-name': nameVal,
                    'fdg-provider': providerVal,
                    'fdg-sort': sortVal
                });
                const added = Array.isArray(data?.games) ? data.games.length : 0;
                if (added > 0) {
                    renderService.renderGames(gamesGrid, data.games, false)
                }

                const newViewed = Math.min((viewed + perPage), totalAll);
                totalMeta.attr('data-viewed-games', newViewed);
                totalMeta.find('.fdgames-viewed').text(newViewed);

                const total = typeof data?.meta?.total === 'number' ? data.meta.total : totalAll;
                if (typeof total === 'number') {
                    if (newViewed >= total) {
                        loadMoreButton.hide();
                    }
                }

                browserUrlService.setQS('page', nextPage);
                browserUrlService.updateURL(true);
            } catch (err) {
                console.error('Failed to load games:', err);
            } finally {
                loadMoreButton.attr('disabled', false);
            }
        });
    }

    function filterFormInit() {
        const form = document.getElementById('fdgames-filters-form');
        const nameInp = document.getElementById('fdgames-name');
        const provider = document.getElementById('fdgames-provider');
        const sortSel = document.getElementById('fdgames-sort');

        if (!form) return;

        function cleanAndSubmit() {
            const toDisable = [];
            for (const el of form.elements) {
                if (!el.name) continue;
                if ((el.value || '').trim() === '') {
                    el.disabled = true;
                    toDisable.push(el);
                }
            }
            form.submit();
            toDisable.forEach(el => (el.disabled = false));
        }

        if (provider) provider.addEventListener('change', cleanAndSubmit);
        if (sortSel) sortSel.addEventListener('change', cleanAndSubmit);

        if (nameInp) {
            nameInp.addEventListener('keydown', function (e) {
                if (e.key === 'Enter') {
                    e.preventDefault();
                    cleanAndSubmit();
                }
            });
        }
    }

})(jQuery)

