<?php

namespace FreedemoGames;

use WP_Post;
use FreedemoGames\Core\Render;
use FreedemoGames\Admin\AdminController;
use FreedemoGames\Admin\PostCustomType\Types\FreedemoPostGameType;

/**
 * Plugin settings
 *
 * @return mixed|void
 */
function fdg_settings()
{
    $settings = get_option(FREEDEMOGAMES_SETTINGS_KEY);

    return apply_filters('fdgames/settings', $settings);
}

function fdg_setting($key, $default = false, string $option = FREEDEMOGAMES_SETTINGS_KEY)
{
    $options = get_option($option, false);

    if (is_array($options) && isset($options[$key])) {
        return apply_filters('fdgames/settings_' . $key, $options[$key]);
    } else {
        return $default;
    }
}

function fdg_define_current_page(): string
{
    return isset($_GET['page']) ? sanitize_key($_GET['page']) : '';
}

function fdg_is_settings_page(): bool
{
    return fdg_define_current_page() === AdminController::MAIN_PLUGIN_SETTING_SLUG;
}

function fdg_current_page(): int
{
    if (isset($_GET['fdg-page'])) {
        return absint($_GET['fdg-page']);
    }

    return 1;
}

/**
 * @param $post_id
 * @param $return_no_img
 *
 * @return string
 */
function fdg_img_url($post_id = null, $return_no_img = true): string
{
    if (!$post_id) {
        $post_id = get_the_ID();
    }
    if (has_post_thumbnail($post_id)) {
        $img = get_the_post_thumbnail_url($post_id);
    } else {
        $img = get_post_meta($post_id, 'fdg_img', true);
    }
    if (!$img && $return_no_img === true) {
        $img = fdg_img_url_no_image();
    }

    return $img;
}

function fdg_img_url_no_image(): string
{
    return FREEDEMOGAMES_PLUGIN_URL . 'assets/public/img/no-image.png';
}

function fdg_game_link($atts = [], $content = '')
{
    return Render::render_public(
        'templates/partials/general',
        'game_link',
        [
            'atts'    => $atts,
            'content' => $content,
        ]
    );
}

function fdg_main_admin_settings_url(): string
{
    return admin_url(sprintf('admin.php?page=%s', AdminController::MAIN_PLUGIN_SETTING_SLUG));
}

function fdg_current_user_can()
{
    $capability = fdg_get_manage_capability();

    return apply_filters('fdgames/current_user_can', current_user_can($capability), $capability);
}

function fdg_get_manage_capability()
{
    return apply_filters('fdgames/manage_capability', 'manage_options');
}

function fdg_get_widget_src(): string
{
    return fdg_setting('slots_widget_src', FREEDEMOGAMES_WIDGET_SRC);
}

function fdg_get_partner_token(): string
{
    return (string)fdg_setting('token');
}

function fdg_get_powered_by_text(): string
{
    return 'FreeDemo.Games';
}

function fdg_get_game_link(string $game_slug): string
{
    $partner_token = fdg_get_partner_token();

    return sprintf('https://freedemo.games/en/slots/%s?token=%s', $game_slug, $partner_token);
}

function fdg_get_powered_by_logo(): string
{
    return '<img src="' . FREEDEMOGAMES_PLUGIN_URL . 'assets/public/img/freedemo-games-logo.png' . '" ' .
           'alt="' . fdg_get_powered_by_text() . '" class="freedemo-games-powered-img"/>';
}

/**
 * @param $source_id
 *
 * @return WP_Post|null
 */
function fdg_get_game_by_game_id($source_id): ?WP_Post
{
    return get_posts([
        'post_type'   => FreedemoPostGameType::TYPE_NAME,
        'meta_key'    => 'fdg_game_id',
        'meta_value'  => (string)$source_id,
        'numberposts' => 1,
    ])[0] ?? null;
}

/**
 * @param $source_id
 *
 * @return WP_Post|null
 */
function fdg_get_game_by_slug(string $slug): ?WP_Post
{
    return get_posts([
        'post_type'   => FreedemoPostGameType::TYPE_NAME,
        'meta_key'    => 'fdg_slug',
        'meta_value'  => (string)$slug,
        'numberposts' => 1,
    ])[0] ?? null;
}
