<?php

namespace FreedemoGames\Admin;

use FreedemoGames\Core\Render;
use FreedemoGames\Core\ShortCodes;
use FreedemoGames\Admin\PostCustomType\Types\FreedemoTypeTaxonomy;
use FreedemoGames\Admin\PostCustomType\Types\FreedemoPostGameType;
use FreedemoGames\Admin\PostCustomType\Types\FreedemoProviderTaxonomy;

use function FreedemoGames\fdg_is_settings_page;

/**
 * Class AdminController
 *
 * @package FreedemoGames\Admin
 */
class AdminController
{
    const MAIN_PLUGIN_SETTING_SLUG = 'fdgames-settings';

    public function __construct()
    {
    }

    public function init(): void
    {
        global $pagenow;
        add_action('admin_menu', [$this, 'add_settings_menu'], 8);
        add_action('admin_notices', [$this, 'header_branding'], 1);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_scripts']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_styles']);

        $fdg_post_type = FreedemoPostGameType::TYPE_NAME;
        add_filter("manage_" . $fdg_post_type . "_posts_columns", [$this, 'set_custom_columns']);
        add_action("manage_" . $fdg_post_type . "_posts_custom_column", [$this, 'custom_columns'], 10, 2);
        add_filter("manage_edit-" . $fdg_post_type . "_sortable_columns", [$this, 'determinate_sortable_columns']);

        add_action('restrict_manage_posts', [$this, 'filter_posts_by_taxonomy']);
    }

    public function add_settings_menu(): void
    {
        $main_menu_hook = add_menu_page(
            'Freedemo Games',
            'Freedemo Games',
            apply_filters('fdgames/settings_page_role', 'manage_options'),
            self::MAIN_PLUGIN_SETTING_SLUG,
            [$this, 'do_admin_page_action'],
            'dashicons-superhero-alt'
        );
        add_action("load-{$main_menu_hook}", [self::class, 'load_setting_page_hooks']);

        add_submenu_page(
            self::MAIN_PLUGIN_SETTING_SLUG,
            'Settings',
            'Settings',
            apply_filters('fdgames/settings_page_role', 'manage_options'),
            self::MAIN_PLUGIN_SETTING_SLUG,
            [$this, 'do_admin_page_action'],
        );

        add_submenu_page(
            self::MAIN_PLUGIN_SETTING_SLUG,
            'Types',
            'Types',
            apply_filters('fdgames/settings_page_role', 'manage_options'),
            sprintf(
                'edit-tags.php?taxonomy=%s&post_type=%s',
                FreedemoTypeTaxonomy::TYPE_NAME,
                FreedemoPostGameType::TYPE_NAME
            ),
        );
    }

    function determinate_sortable_columns($columns)
    {
        $provider                      = FreedemoProviderTaxonomy::TYPE_NAME;
        $type                          = FreedemoTypeTaxonomy::TYPE_NAME;
        $columns["taxonomy-$provider"] = $provider;
        $columns["taxonomy-$type"]     = $type;

        return $columns;
    }

    public static function load_setting_page_hooks()
    {
    }

    public static function header_branding(): void
    {
        if (fdg_is_settings_page()) {
            echo Render::render_admin('/partials/layout', 'header');
        }
    }

    public function do_admin_page_action(): void
    {
        do_action(Settings::MAIN_ACTION_NAME);
    }

    public function enqueue_styles(): void
    {
        if (fdg_is_settings_page()) {
            wp_enqueue_style(
                'freedemo-games-admin-css',
                FREEDEMOGAMES_PLUGIN_URL . 'assets/admin/css/freedemo-games-admin.css',
                [],
                FREEDEMOGAMES_VERSION
            );
            wp_enqueue_style(
                'freedemo-games-bootstrap-admin-css',
                FREEDEMOGAMES_PLUGIN_URL . 'assets/admin/css/bootstrap.min.css',
                [],
                FREEDEMOGAMES_VERSION
            );
        }
    }

    public function enqueue_scripts(): void
    {
        if (fdg_is_settings_page()) {
            wp_enqueue_script(
                'freedemo-games-admin',
                FREEDEMOGAMES_PLUGIN_URL . 'assets/admin/js/freedemo-games-admin.js',
                ['jquery'],
                FREEDEMOGAMES_VERSION,
                true
            );
        }
        wp_enqueue_script(
            'bootstrap-js',
            FREEDEMOGAMES_PLUGIN_URL . 'assets/admin/js/bootstrap.bundle.min.js',
            ['jquery'],
            null,
            true
        );

        wp_localize_script('freedemo-games-admin', 'fdgames_params', [
            'ajax_url' => admin_url('admin-ajax.php'),
        ]);
    }

    public function set_custom_columns($columns): array
    {
        $new_column = [];
        unset($columns['author']);
        unset($columns['comments']);

        foreach ($columns as $key => $value) {
            if ($key == 'date') {
                $new_column['fdgames_shortcode'] = __('Embed slot', 'freedemo-games');
            }
            $new_column[$key] = $value;
        }

        return $new_column;
    }

    function custom_columns($column, $post_id): void
    {
        if ($column == 'fdgames_shortcode') {
            echo sprintf('[%s fdpost_id="%s"]', ShortCodes::SHORT_CODE_GAME_SINGLE, $post_id);
        }
    }

    public function filter_posts_by_taxonomy()
    {
        global $typenow;

        if ($typenow == FreedemoPostGameType::TYPE_NAME) {
            $terms = get_terms(FreedemoProviderTaxonomy::TYPE_NAME);
            $this->printFilterDropdown(
                $terms,
                FreedemoProviderTaxonomy::TYPE_NAME,
                __('All Providers', 'freedemo-games')
            );
            $terms = get_terms(FreedemoTypeTaxonomy::TYPE_NAME);
            $this->printFilterDropdown($terms, FreedemoTypeTaxonomy::TYPE_NAME, __('All Types', 'freedemo-games'));
        }
    }

    private function printFilterDropdown($terms, $taxonomy, $all)
    {
        if (!empty($terms)) {
            echo '<select name="' . $taxonomy . '" id="' . $taxonomy . '" class="postform">';
            echo '<option value="">' . $all . '</option>';

            foreach ($terms as $term) {
                $selected = isset($_GET[$taxonomy]) && $_GET[$taxonomy] == $term->slug ? ' selected="selected"' : '';
                echo '<option value="' . $term->slug . '"' . $selected . '>' . $term->name . '</option>';
            }

            echo '</select>';
        }
    }

}
