<?php

namespace FreedemoGames\Admin;

use FreedemoGames\Core\Render;

use function FreedemoGames\fdg_is_settings_page;
use function FreedemoGames\fdg_current_user_can;

/**
 * Class Settings
 *
 * @package FreedemoGames\Admin
 */
class Settings
{
    const MAIN_ACTION_NAME = 'freedemo-games_settings_page';

    public string $view;

    public function __construct()
    {
    }

    public function init(): void
    {
        if (fdg_is_settings_page()) {
            // important register function save_settings
            add_action('admin_init', [$this, 'save_settings']);

            $this->view = isset($_GET['view']) ? sanitize_key($_GET['view']) : 'general';

            add_action(self::MAIN_ACTION_NAME, [$this, 'output']);

            do_action('freedemo-games_settings_init');
        }
    }

    public function check_can_save()
    {
        if (!isset($_POST['fdgames-settings-submit'])) {
            return false;
        }

        if (!wp_verify_nonce($_POST['nonce'], 'fdgames-settings-nonce')) {
            return false;
        }

        if (!fdg_current_user_can()) {
            return false;
        }

        if (empty($_POST['view'])) {
            return false;
        }

        return true;
    }

    public function save_settings()
    {
        if (!$this->check_can_save()) {
            return;
        }

        $fields   = SettingsFields::get_predefined_settings($_POST['view']);
        $settings = get_option(FREEDEMOGAMES_SETTINGS_KEY, []);

        if (empty($fields) || !is_array($fields)) {
            return;
        }
        // refresh permalinks in case of slug change
        $shouldUpdateUrl = false;

        $exclude = apply_filters('fdgames/settings_exclude_type', [
            'content',
            'button',
            'ajax_button',
        ]);

        foreach ($fields as $id => $field) {

            if (empty($field['type']) || in_array($field['type'], $exclude, true)) {
                continue;
            }

            $value      = $_POST[$id] ?? false;
            $value_prev = $settings[$id] ?? false;

            if (!empty($field['filter']) && function_exists($field['filter'])) {
                $value = call_user_func($field['filter'], $value, $id, $field, $value_prev);
            } else {
                switch ($field['type']) {
                    case 'checkbox':
                        $value = (bool)trim($value);
                        break;
                    case 'image':
                        $value = esc_url_raw(trim($value));
                        break;
                    case 'text':
                    case 'radio':
                    case 'select':
                        if (is_array($value)) {
                            $value = array_map('sanitize_text_field', $value);
                        } else {
                            $value = sanitize_text_field(trim($value));
                        }
                        break;
                    default:
                        $value = apply_filters(
                            'fdgames_sanitize_field_' . $field['type'],
                            $value,
                            $id,
                            $field,
                            $value_prev
                        );
                        break;
                }
            }

            if ('slots-slug' == $id && (isset($settings[$id]) && $settings[$id] != $value)) {
                $value = sanitize_title($value);
                $shouldUpdateUrl = true;
            }
            $settings[$id] = $value;
        }

        update_option(FREEDEMOGAMES_SETTINGS_KEY, $settings);

        if ($shouldUpdateUrl) {
            update_option('fdg_needs_flush', 1);
        }

        Notices::success(esc_html__('Settings were successfully saved.', 'freedemo-games'));
    }

    public static function should_update_url(): void
    {
        if (get_option('fdg_needs_flush')) {
            flush_rewrite_rules();
            delete_option('fdg_needs_flush');
        }
    }


    public function get_settings_fields($view = '')
    {
        $fields   = [];
        $settings = SettingsFields::get_predefined_settings($view);

        foreach ($settings as $id => $params) {
            if (isset($params['type']) && $params['type']) {
                $fields[$id] = SettingsFields::get_settings_field($params['type'], $params);
            }
        }

        return apply_filters('fdgames/settings_fields', $fields, $view);
    }

    public function get_tabs()
    {
        $tabs = [
            'general' => [
                'name'   => esc_html__('General', 'freedemo-games'),
                'form'   => true,
                'submit' => esc_html__('Save Settings', 'freedemo-games'),
            ],
        ];

        return apply_filters('fdgames/settings_tabs', $tabs);
    }

    public function output()
    {
        $tabs   = $this->get_tabs();
        $fields = $this->get_settings_fields($this->view);

        echo Render::render_admin('/partials/settings', 'content-wrap', [
            'tabs'   => $tabs,
            'fields' => $fields,
            'view'   => $this->view,
        ]);
    }

}

