<?php

namespace FreedemoGames\Core;

use FreedemoGames\Admin\PostCustomType\Types\FreedemoPostGameType;
use FreedemoGames\Admin\PostCustomType\Types\FreedemoTypeTaxonomy;
use FreedemoGames\Admin\PostCustomType\Types\FreedemoProviderTaxonomy;

use function FreedemoGames\fdg_get_game_by_game_id;

/**
 * Class GamesImporterService
 *
 * @package FreedemoGames\Core
 */
class GamesImporterService
{
    private static $provider_terms = [];
    const TRANSIENT_LOCK_KEY = 'fdg_import_games_lock';

    public function __construct()
    {
        update_option('fdg_admin_id', get_current_user_id());
    }

    public function import(int $startPage = 1)
    {
        if (get_transient(self::TRANSIENT_LOCK_KEY)) {
            return [
                'status' => 'warning',
                'msg'    => 'Import already running. This process may take approximately 15 minutes.',
            ];
        }
        set_transient(self::TRANSIENT_LOCK_KEY, 1, 60 * 2); // lock other imports

        add_filter('do_rocket_generate_caching_files', '__return_false');
        add_filter('rocket_generate_advanced_cache_file', '__return_false');
        add_filter('rocket_disable_htaccess', '__return_false');
        add_filter('pre_get_rocket_option_sitemap_preload', '__return_zero');
        add_filter('pre_get_rocket_option_sitemap_preload_url_crawl', '__return_zero');
        add_filter('pre_get_rocket_option_sitemaps', '__return_zero');
        add_filter('pre_get_rocket_option_manual_preload', '__return_zero');


        $page = max(1, $startPage);;
        $limit       = 250;
        $totalDone   = 0;
        $seenSame    = 0;     // stuck on one the same page counter
        $lastPageNum = null;
        $maxLoops    = 10000; // hard fuse
        try {
            for ($i = 0; $i < $maxLoops; $i++) {
                $args = [
                    'page'  => $page,
                    'limit' => $limit,
                ];

                $games  = ApiClient::get_games($args);
                if($games['success'] === false) {
                    return [
                        'status' => 'error',
                        'msg'    => $games['status'],
                    ];
                }

                $result = $this->insert_games($games, $page);
                if ($result['status'] === 'error') {
                    return $result;
                }
                $totalDone += (int)($result['processed'] ?? 0);

                $meta      = $games['meta'] ?? [];
                $currPage  = (int)($meta['page'] ?? $page);
                $currLimit = (int)($meta['limit'] ?? $limit);
                $count     = (int)($meta['count'] ?? count($games['data'] ?? []));
                $total     = (int)($meta['total'] ?? 0);
                $pages     = (int)($meta['pages'] ?? ($currLimit > 0 && $total > 0 ? (int)ceil(
                    $total / $currLimit
                ) : 0));
                $nextMeta  = isset($meta['next']) ? (int)$meta['next'] : null;

                $noMoreByPages = ($pages > 0 && $currPage >= $pages);
                $noMoreByCount = ($pages === 0 && $count < $currLimit);
                $noData        = ($count === 0);

                if ($noMoreByPages || $noMoreByCount || $noData) {
                    return [
                        'status'    => 'success',
                        'msg'       => 'Games import completed.',
                        'processed' => $totalDone,
                    ];
                }

                $nextPage = $nextMeta ?: ($currPage + 1);

                // stuck on one the same page detector
                if ($lastPageNum !== null && $nextPage === $lastPageNum) {
                    $seenSame++;
                } else {
                    $seenSame = 0;
                }
                if ($seenSame >= 3) {
                    // on 3 tries left the loop
                    return [
                        'status'    => 'error',
                        'msg'       => "Stuck on page {$nextPage}. Aborting to avoid infinite loop.",
                        'processed' => $totalDone,
                    ];
                }

                $lastPageNum = $page = $nextPage;
                $limit       = $currLimit;

                if (($page % 2) === 0) {
                    set_transient(self::TRANSIENT_LOCK_KEY, 1, 60 * 2);  // heartbeat
                }
            }

            return [
                'status'    => 'success',
                'msg'       => 'Import completed.',
                'processed' => $totalDone,
            ];
        } catch (\Throwable $e) {
            return [
                'status'    => 'error',
                'msg'       => $e->getMessage(),
                'processed' => $totalDone,
            ];
        } finally {
            delete_transient(self::TRANSIENT_LOCK_KEY);
            flush_rewrite_rules(false);

            remove_filter('do_rocket_generate_caching_files', '__return_false');
            remove_filter('rocket_generate_advanced_cache_file', '__return_false');
            remove_filter('rocket_disable_htaccess', '__return_false');
            remove_filter('pre_get_rocket_option_sitemap_preload', '__return_zero');
            remove_filter('pre_get_rocket_option_sitemap_preload_url_crawl', '__return_zero');
            remove_filter('pre_get_rocket_option_sitemaps', '__return_zero');
            remove_filter('pre_get_rocket_option_manual_preload', '__return_zero');
        }
    }

    private function insert_games(array $games, int $page): array
    {
        $didDefer  = false;
        $didKses   = false;
        $processed = 0;

        try {
            if (!empty($games['error'])) {
                return [
                    'status' => 'error',
                    'msg'    => $games['error'],
                ];
            }
            wp_defer_term_counting(true);
            wp_defer_comment_counting(true);
            $didDefer = true;

            kses_remove_filters();
            $didKses = true;

            $author = get_option('fdg_admin_id');

            if (empty($games['data'])) {
                return [
                    'status'    => 'success',
                    'msg'       => 'No more games',
                    'processed' => 0,
                    'next_page' => null,
                ];
            }

            foreach ($games['data'] as $game) {
                if (empty($game['game_id'])) {
                    continue;
                }

                $gmt        = date('Y-m-d H:i:s');
                $game_attrs = [
                    'release'                => $game['release'] ?? '',
                    'payline'                => $game['payline'] ?? '',
                    'rtp'                    => $game['rtp'] ?? '',
                    'volatility'             => $game['volatility'] ?? '',
                    'currencies'             => $game['currencies'] ?? '',
                    'languages'              => $game['languages'] ?? '',
                    'land_based'             => $game['land_based'] ?? '',
                    'markets'                => $game['markets'] ?? '',
                    'progressive'            => $game['progressive'] ?? '',
                    'min_bet'                => $game['min_bet'] ?? '',
                    'max_bet'                => $game['max_bet'] ?? '',
                    'max_win_per_spin'       => $game['max_win_per_spin'] ?? '',
                    'bonus_buy'              => $game['bonus_buy'] ?? '',
                    'autoplay'               => $game['autoplay'] ?? '',
                    'quickspin'              => $game['quickspin'] ?? '',
                    'increasing_multipliers' => $game['increasing_multipliers'] ?? '',
                    'orientation'            => $game['orientation'] ?? '',
                    'restrictions'           => $game['restrictions'] ?? '',
                ];

                $images = [];

                $my_post = [
                    'post_title'    => $game['name'],
                    'post_content'  => ShortCodes::create_content_short_code_game(),
                    'post_status'   => 'publish',
                    'post_date_gmt' => $gmt,
                    'post_date'     => get_gmt_from_date($gmt),
                    'post_excerpt'  => $game['name'],
                    'post_type'     => FreedemoPostGameType::TYPE_NAME,
                    'post_author'   => $author,
                    'meta_input'    => [
                        'fdg_id'                => $game['id'],
                        'fdg_game_id'           => $game['game_id'],
                        'fdg_vendor_id'         => $game['vendor_id'],
                        'fdg_slug'              => $game['slug'],
                        'fdg_visibility_status' => $game['visibility_status'],
                        'fdg_start_date'        => $game['start_date'],
                        'fdg_wide_release'      => $game['wide_release'],
                        'fdg_end_date'          => $game['end_date'],
                        'fdg_game_attrs'        => $game_attrs,
                    ],
                ];

                if (!empty($game['images'])) {
                    $images = ImageService::group_images_by_key($game['images'], ImageService::ORIENTATION);
                }
                $img_rectangle = $images['rectangle'][0]['full_url'] ?? null;
                $img_square    = $images['square'][0]['full_url'] ?? null;

                $my_post['meta_input'] = array_merge($my_post['meta_input'], [
                    'fdg_img_rectangle' => $img_rectangle,
                    'fdg_img_square'    => $img_square,
                    'fdg_img'           => $img_rectangle ?: $img_square,
                ]);


                if (!GameValidatorService::are_there_required_fields($my_post['meta_input'], false)) {
                    continue;
                }

                $existing_game = fdg_get_game_by_game_id($game['game_id']);
                if ($existing_game) {
                    $my_post['ID'] = $existing_game->ID;
                    $pid           = wp_update_post(apply_filters('fdgames/posts_args', $my_post), true);
                } else {
                    $pid = wp_insert_post(apply_filters('fdgames/posts_args', $my_post), true);
                }

                $processed++;

                if (is_wp_error($pid)) {
                    return ['status' => 'error', 'msg' => $pid->get_error_message()];
                }

                if (is_numeric($pid)) {
                    if (!empty($game['vendor_id'])) {
                        $vendor_id = (int)$game['vendor_id'];

                        $providerTermId = self::$provider_terms[$vendor_id] ?? null;
                        if (!$providerTermId) {
                            $terms = get_terms([
                                'taxonomy'   => FreedemoProviderTaxonomy::TYPE_NAME,
                                'hide_empty' => false,
                                'number'     => 1,
                                'fields'     => 'ids',
                                'meta_query' => [
                                    [
                                        'key'   => 'fdgame_vid',
                                        'value' => (int)$game['vendor_id'],
                                    ],
                                ],
                            ]);

                            if (!is_wp_error($terms) && !empty($terms)) {
                                $providerTermId = (int)$terms[0];
                            }
                            self::$provider_terms[$vendor_id] = $providerTermId;
                        }

                        if ($providerTermId) {
                            wp_set_object_terms($pid, $providerTermId, FreedemoProviderTaxonomy::TYPE_NAME, false);
                        }
                    }

                    if (!empty(isset($game['type']))) {
                        wp_set_object_terms($pid, [$game['type']], FreedemoTypeTaxonomy::TYPE_NAME, true);
                    }
                }
            }

            return [
                'status'    => 'success',
                'msg'       => 'Page processed',
                'processed' => $processed,
                'next_page' => $page + 1,
            ];
        } catch (\Exception $exception) {
            return [
                'status' => 'error',
                'msg'    => $exception->getMessage(),
            ];
        } finally {
            if ($didKses) {
                kses_init_filters();
            }
            if ($didDefer) {
                wp_defer_term_counting(false);
                wp_defer_comment_counting(false);
            }
        }
    }
}
