<?php

namespace FreedemoGames\Core\GamesSource;

use WP_Query;
use FreedemoGames\Core\GamesSource\Entities\GameEntity;
use FreedemoGames\Core\GamesSource\Dto\GamesCriteriaDto;
use FreedemoGames\Core\GamesSource\Entities\ProviderEntity;
use FreedemoGames\Core\GamesSource\Entities\Vo\ProviderImage;
use FreedemoGames\Core\GamesSource\Collection\GamesCollectionMeta;
use FreedemoGames\Admin\PostCustomType\Types\FreedemoTypeTaxonomy;
use FreedemoGames\Admin\PostCustomType\Types\FreedemoPostGameType;
use FreedemoGames\Core\GamesSource\Collection\GamesEntityCollection;
use FreedemoGames\Admin\PostCustomType\Types\FreedemoFiltersTaxonomy;
use FreedemoGames\Admin\PostCustomType\Types\FreedemoProviderTaxonomy;
use FreedemoGames\Core\GamesSource\Exceptions\GameDoesntExistException;
use FreedemoGames\Core\GamesSource\Collection\ProviderEntityCollection;
use FreedemoGames\Core\GamesSource\Exceptions\InvalidGameArgumentsException;

use function FreedemoGames\fdg_get_game_by_slug;
use function FreedemoGames\fdg_get_game_by_game_id;

/**
 * Class RepositoryLocal
 *
 * @package FreedemoGames\Core\GamesSource
 */
class RepositoryLocal implements RepositoryInterface
{
    /**
     * @param GamesCriteriaDto $criteria
     * @param array            $args
     *
     * @return GamesEntityCollection
     */
    public function get_games(
        GamesCriteriaDto $criteria,
        array $args = []
    ): GamesEntityCollection {
        $query_params = [
            'post_type'      => FreedemoPostGameType::TYPE_NAME,
            'posts_per_page' => !empty($args['per_page']) ? absint($args['per_page']) : 40,
            //'paged'          => fdg_current_page(),
            'offset'         => $args['offset'],
            'post_status'    => 'publish',
            'orderby'        => 'ID',
            'order'          => 'DESC',
            'no_found_rows'  => false,
            'meta_query'     => [
                'relation' => 'AND',
                [
                    'key'     => 'fdg_visibility_status',
                    'value'   => 'visible',
                    'compare' => '=',
                ],
            ],
        ];

        if ($criteria->sort == 'az' || $criteria->sort == 'za') {
            $query_params['orderby'] = 'title';
            $query_params['order']   = $criteria->sort == 'az' ? 'asc' : 'desc';
        }

        if (!empty($args['order_by'])) {
            $query_params['orderby'] = $args['order_by'];
        }

        if (!empty($args['order'])) {
            $query_params['order'] = $args['order'];
        }

        if ($criteria->type) {
            $query_params['tax_query'][] =
                [
                    'taxonomy' => FreedemoTypeTaxonomy::TYPE_NAME,
                    'field'    => 'slug',
                    'terms'    => is_array($criteria->type)
                        ? $criteria->type
                        : array_map(
                            'trim',
                            explode(',', $criteria->type)
                        ),
                ];
        }

        if ($criteria->provider) {
            $query_params['tax_query'][] =
                [
                    'taxonomy' => FreedemoProviderTaxonomy::TYPE_NAME,
                    'field'    => 'slug',
                    'terms'    => is_array($criteria->provider)
                        ? $criteria->provider
                        : array_map(
                            'trim',
                            explode(',', $criteria->provider)
                        ),
                ];
        }

        if ($criteria->filters) {
            $query_params['tax_query'][] =
                [
                    'taxonomy' => FreedemoFiltersTaxonomy::TYPE_NAME,
                    'field'    => 'slug',
                    'terms'    => is_array($criteria->filters)
                        ? $criteria->filters
                        : array_map(
                            'trim',
                            explode(',', $criteria->filters)
                        ),
                ];
        }

        if ($criteria->search) {
            $query_params['s'] = esc_sql($criteria->search);
        }

        $args     = array_filter($args, static fn($v) => $v !== '' && $v !== null);
        $args     = wp_parse_args($args, $query_params);
        $wp_query = new WP_Query($args);
        $games    = [];
        while ($wp_query->have_posts()) {
            $wp_query->the_post();
            try {
                $games[] = $this->build_game_entity(get_the_ID());
            } catch (InvalidGameArgumentsException $e) {
                continue;
            }
        }
        $games_mata = new GamesCollectionMeta($wp_query->found_posts);

        return new GamesEntityCollection($games, $games_mata);
    }

    /**
     * @param array $atts
     *
     * @return GameEntity
     * @throws GameDoesntExistException
     * @throws InvalidGameArgumentsException
     */
    public function get_game(array $atts = []): GameEntity
    {
        $game = null;
        if (!empty($atts['fdgames_id'])) {
            $game = fdg_get_game_by_game_id($atts['fdgames_id']);
        }

        if (!$game && !empty($atts['fdgames_slug'])) {
            $game = fdg_get_game_by_slug($atts['fdgames_slug']);
        }

        if (!$game) {
            $game = get_post($atts['fdpost_id'] ?? null);
        }

        if (is_null($game)) {
            throw new GameDoesntExistException("The game doesn't exist! Please make sure the post ID is correct.");
        }

        return $this->build_game_entity($game->ID, $atts);
    }

    /**
     * @param int   $post_id
     * @param array $atts
     *
     * @return GameEntity
     * @throws InvalidGameArgumentsException
     */
    private function build_game_entity(int $post_id, array $atts = []): GameEntity
    {
        $game     = get_post($post_id);
        $meta_raw = get_post_meta($game->ID);
        $meta     = array_map(static fn($v) => $v[0] ?? null, $meta_raw);

        return GameBuilder::build_game($game->ID, $game->post_title, $meta, $atts);
    }

    /**
     * @param array $args
     *
     * @return ProviderEntityCollection
     */
    public function get_providers(array $args = []): ProviderEntityCollection
    {
        $terms = get_terms([
            'taxonomy'   => FreedemoProviderTaxonomy::TYPE_NAME,
            'hide_empty' => true,
        ]);

        $providers = [];

        if (!is_wp_error($terms) && !empty($terms)) {
            foreach ($terms as $term) {
                $meta = get_term_meta($term->term_id);

                $providers[] = new ProviderEntity(
                    $term->term_id,
                    $meta['fdgame_vid'][0],
                    $term->name,
                    $term->slug,
                    $meta['fdgame_plink'][0] ?? null,
                    isset($meta['provider_image'][0]) ? new ProviderImage($meta['provider_image'][0]) : null,
                    $meta['fdgame_pvid'][0] ?? null,
                    $meta['fdgame_pyear_start'][0] ?? null,
                    $meta['fdgame_pyear_end'][0] ?? null,
                );
            }
        }

        return new ProviderEntityCollection($providers);
    }
}
