<?php

namespace FreedemoGames\Core;

/**
 * Class HttpClient
 *
 * @package FreedemoGames\Core
 */
class HttpClient
{
    private string $host;
    private array  $default_headers = [];
    private bool   $debug;

    /**
     * @param string $host
     * @param array  $default_headers
     * @param bool   $debug
     */
    public function __construct(string $host, array $default_headers = [], bool $debug = false)
    {
        $this->host            = rtrim($host, '/');
        $this->default_headers = $default_headers;
        $this->debug           = $debug;
    }

    public function get_host(): string
    {
        return $this->host;
    }

    public function get(string $endpoint, array $query = [], array $headers = []): array
    {
        $url = $this->build_url($endpoint, $query);

        return $this->request('GET', $url, [], $headers);
    }

    public function post(string $endpoint, array $body = [], array $headers = []): array
    {
        $url = $this->build_url($endpoint);

        return $this->request('POST', $url, $body, $headers);
    }

    private function build_url(string $endpoint, array $query = []): string
    {
        $url = $this->host . '/' . ltrim($endpoint, '/');
        if (!empty($query)) {
            $url .= '?' . http_build_query($query);
        }

        return $url;
    }

    private function request(
        string $method,
        string $url,
        array $body = [],
        array $headers = [],
        int $timeout = 10
    ): array {
        $args = [
            'method'  => $method,
            'headers' => array_merge($this->default_headers, $headers),
            'timeout' => $timeout,
        ];

        if (!empty($body)) {
            $args['headers']['Content-Type'] = 'application/json';
            $args['body']                    = wp_json_encode($body);
        }

        $response = wp_remote_request($url, $args);

        if ($this->debug) {
            $this->debug_log([
                'request'  => [
                    'method'  => $method,
                    'url'     => $url,
                    'body'    => $body,
                    'headers' => $headers,
                ],
                'response' => $response,
            ]);
        }

        if (is_wp_error($response)) {
            return [
                'success' => false,
                'status'  => $response->get_error_code(),
                'error'   => $response->get_error_message(),
                'data'    => null,
                'meta'    => null,
            ];
        }

        $status_code = wp_remote_retrieve_response_code($response);
        $raw_body    = wp_remote_retrieve_body($response);
        $data        = $raw_body;
        $meta        = [];

        $decoded = json_decode($raw_body, true);

        if (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) {
            $data = $decoded['data'] ?? $decoded;
            $meta = $decoded['meta'] ?? [];
        }

        return [
            'success' => $status_code >= 200 && $status_code < 300,
            'status'  => $status_code,
            'error'   => false,
            'data'    => $data,
            'meta'    => $meta,
        ];
    }


    private function debug_log(array $data): void
    {
        $json = wp_json_encode(
            $data,
            JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE
        );

        $file = FREEDEMOGAMES_PLUGIN_DIR . '/fdg-http.log';
        file_put_contents(
            $file,
            "[" . gmdate('Y-m-d H:i:s') . "] " . $json . "\n",
            FILE_APPEND | LOCK_EX
        );
    }
}
