<?php

namespace FreedemoGames\Core;

use FreedemoGames\Admin\Updater;
use FreedemoGames\Admin\Settings;
use FreedemoGames\Admin\AdminController;
use FreedemoGames\Admin\AdminAjaxController;
use FreedemoGames\PublicSide\PublicSideController;
use FreedemoGames\PublicSide\PublicSideAjaxController;
use FreedemoGames\Admin\PostCustomType\CustomTypesManager;
use FreedemoGames\Admin\PostCustomType\Types\FreedemoPostGameType;
use FreedemoGames\Admin\PostCustomType\Types\FreedemoTypeTaxonomy;
use FreedemoGames\Admin\PostCustomType\Types\FreedemoFiltersTaxonomy;
use FreedemoGames\Admin\PostCustomType\Types\FreedemoProviderTaxonomy;

use function FreedemoGames\fdg_get_partner_token;

/**
 * Class Kernel
 *
 * @package FreedemoGames\Core
 */
class Kernel
{
    private static ApiClient $api_client;
    private static ?Kernel   $_instance = null;

    /**
     * @return self|null
     */
    public static function getInstance($is_admin = false): self
    {
        if (is_null(self::$_instance)) {
            self::$_instance = new self($is_admin);
        }

        return self::$_instance;
    }

    /**
     *
     */
    private function __construct($is_admin)
    {
        $this->ini_api_client();
        add_action('init', [$this, 'define_custom_types']);
        add_action('init', [Settings::class, 'should_update_url'], 999);

        if ($is_admin) {
            (new AdminController())->init();
            (new Settings())->init();
            (new AdminAjaxController(
                new GamesImporterService(),
                new ProvidersImporterService(),
            ))->init();
            (new Updater())->init();
        }

        (new PublicSideController())->init();
        (new PublicSideAjaxController())->init();
    }

    public function define_custom_types(): void
    {
        (new CustomTypesManager(...[
            new FreedemoPostGameType(),
            new FreedemoProviderTaxonomy(),
            new FreedemoTypeTaxonomy(),
            new FreedemoFiltersTaxonomy(),
        ]))->init();
    }

    private function ini_api_client(): void
    {
        $api_host = '';
        if (defined('FREEDEMOGAMES_API_HOST') && FREEDEMOGAMES_API_HOST !== '') {
            $api_host = FREEDEMOGAMES_API_HOST;
        }
        $siteUrl = site_url();
        $host    = wp_parse_url($siteUrl, PHP_URL_HOST);
        if (function_exists('idn_to_ascii')) {
            $host = idn_to_ascii($host, 0, INTL_IDNA_VARIANT_UTS46) ?: $host;
        }

        $token            = fdg_get_partner_token();
        self::$api_client = new ApiClient($api_host, [
            'X-Token'              => $token,
            'X-Client-Host'        => $host,
            'X-Freedemo-Games-Tag' => FREEDEMOGAMES_DEFAULT_API_TAG,
        ]);
    }

    public static function get_api_client(): ApiClient
    {
        return self::$api_client;
    }

    /**
     * @return void
     */
    private function __clone()
    {
        //
    }

    /**
     * @return void
     */
    public function __wakeup()
    {
        //
    }
}
