<?php

declare(strict_types = 1);

namespace FreedemoGames\Core;

use FreedemoGames\Admin\PostCustomType\Types\FreedemoProviderTaxonomy;

/**
 * Class ProvidersImporterService
 *
 * @package Core
 */
class ProvidersImporterService
{
    const TRANSIENT_LOCK_KEY = 'fdg_import_vendors_lock';

    public function import(int $startPage = 1): array
    {
        if (get_transient(self::TRANSIENT_LOCK_KEY)) {
            return [
                'status' => 'warning',
                'msg'    => 'Vendors import already running.',
            ];
        }
        set_transient(self::TRANSIENT_LOCK_KEY, 1, 60 * 2); // lock other imports

        add_filter('do_rocket_generate_caching_files', '__return_false');
        add_filter('rocket_generate_advanced_cache_file', '__return_false');
        add_filter('rocket_disable_htaccess', '__return_false');
        add_filter('pre_get_rocket_option_sitemap_preload', '__return_zero');
        add_filter('pre_get_rocket_option_sitemap_preload_url_crawl', '__return_zero');
        add_filter('pre_get_rocket_option_sitemaps', '__return_zero');
        add_filter('pre_get_rocket_option_manual_preload', '__return_zero');

        $page        = max(1, $startPage);
        $limit       = 250;
        $totalDone   = 0;
        $seenSame    = 0;     // stuck on one the same page counter
        $lastPageNum = null;
        $maxLoops    = 5000; // hard fuse

        try {
            for ($i = 0; $i < $maxLoops; $i++) {
                $args = [
                    'page'  => $page,
                    'limit' => $limit,
                ];

                $vendors = ApiClient::get_providers($args);
                if($vendors['success'] === false) {
                    return [
                        'status' => 'error',
                        'msg'    => $vendors['status'],
                    ];
                }
                $result  = $this->insert_vendors($vendors, $page);
                if (($result['status'] ?? null) === 'error') {
                    return $result;
                }
                $totalDone += (int)($result['processed'] ?? 0);

                $meta      = $vendors['meta'] ?? [];
                $currPage  = (int)($meta['page'] ?? $page);
                $currLimit = (int)($meta['limit'] ?? $limit);
                $count     = (int)($meta['count'] ?? count($vendors['data'] ?? []));
                $total     = (int)($meta['total'] ?? 0);
                $pages     = (int)($meta['pages'] ?? ($currLimit > 0 && $total > 0 ? (int)ceil(
                    $total / $currLimit
                ) : 0));
                $nextMeta  = isset($meta['next']) ? (int)$meta['next'] : null;

                $noMoreByPages = ($pages > 0 && $currPage >= $pages);
                $noMoreByCount = ($pages === 0 && $count < $currLimit);
                $noData        = ($count === 0);

                if ($noMoreByPages || $noMoreByCount || $noData) {
                    return [
                        'status'    => 'success',
                        'msg'       => 'Vendors import completed.',
                        'processed' => $totalDone,
                    ];
                }

                $nextPage = $nextMeta ?: ($currPage + 1);

                // stuck on one the same page detector
                if ($lastPageNum !== null && $nextPage === $lastPageNum) {
                    $seenSame++;
                } else {
                    $seenSame = 0;
                }
                if ($seenSame >= 3) {
                    // on 3 tries left the loop
                    return [
                        'status'    => 'error',
                        'msg'       => "Stuck on page {$nextPage}. Aborting to avoid infinite loop.",
                        'processed' => $totalDone,
                    ];
                }

                $lastPageNum = $page = $nextPage;
                $limit       = $currLimit;

                if (($page % 2) === 0) {
                    set_transient(self::TRANSIENT_LOCK_KEY, 1, 60 * 2);  // heartbeat
                }
            }

            return [
                'status'    => 'success',
                'msg'       => 'Import completed (max loops reached).',
                'processed' => $totalDone,
            ];
        } catch (\Throwable $e) {
            return [
                'status'    => 'error',
                'msg'       => $e->getMessage(),
                'processed' => $totalDone,
            ];
        } finally {
            delete_transient(self::TRANSIENT_LOCK_KEY);
            //flush_rewrite_rules(false);

            remove_filter('do_rocket_generate_caching_files', '__return_false');
            remove_filter('rocket_generate_advanced_cache_file', '__return_false');
            remove_filter('rocket_disable_htaccess', '__return_false');
            remove_filter('pre_get_rocket_option_sitemap_preload', '__return_zero');
            remove_filter('pre_get_rocket_option_sitemap_preload_url_crawl', '__return_zero');
            remove_filter('pre_get_rocket_option_sitemaps', '__return_zero');
            remove_filter('pre_get_rocket_option_manual_preload', '__return_zero');
        }
    }

    private function insert_vendors($vendors, int $page): array
    {
        $didDefer  = false;
        $didKses   = false;
        $processed = 0;

        try {
            if (!empty($vendors['error'])) {
                return [
                    'status' => 'error',
                    'msg'    => $vendors['error'],
                ];
            }

            wp_defer_term_counting(true);
            wp_defer_comment_counting(true);
            $didDefer = true;

            kses_remove_filters();
            $didKses = true;

            if (empty($vendors['data'])) {
                return [
                    'status'    => 'success',
                    'msg'       => 'No vendors on this page.',
                    'processed' => 0,
                    'next_page' => null,
                ];
            }

            foreach ($vendors['data'] as $vendor) {
                $terms = get_terms([
                    'taxonomy'   => FreedemoProviderTaxonomy::TYPE_NAME,
                    'hide_empty' => false,
                    'meta_query' => [
                        [
                            'key'   => 'fdgame_vid',
                            'value' => (int)$vendor['vendor_id'],
                        ],
                    ],
                    'number'     => 1,
                ]);

                $existing_vendor = (!is_wp_error($terms) && !empty($terms)) ? $terms[0] : null;

                if ($existing_vendor && !is_wp_error($existing_vendor)) {
                    $term_id = $existing_vendor->term_id;

                    $upd = wp_update_term(
                        $term_id,
                        FreedemoProviderTaxonomy::TYPE_NAME,
                        [
                            'name' => $vendor['name'],
                            'slug' => $vendor['slug'],
                        ]
                    );

                    if (is_wp_error($upd)) {
                        return [
                            'status' => 'error',
                            'msg'    => $upd->get_error_message(),
                        ];
                    }
                } else {
                    $ins = wp_insert_term(
                        $vendor['name'],
                        FreedemoProviderTaxonomy::TYPE_NAME,
                        [
                            'description' => $vendor['name'] . ' Games',
                            'slug'        => $vendor['slug'],
                        ]
                    );

                    if (is_wp_error($ins)) {
                        if ($ins->get_error_code() === 'term_exists') {
                            $term_id = (int)$ins->get_error_data();
                        } else {
                            return [
                                'status' => 'error',
                                'msg'    => $ins->get_error_message(),
                            ];
                        }
                    } else {
                        $term_id = (int)$ins['term_id'];
                    }
                }

                update_term_meta($term_id, 'fdgame_vid', $vendor['vendor_id']);
                if ($parent_id = (int)$vendor['parent_id']) {
                    update_term_meta($term_id, 'fdgame_pvid', $parent_id);
                }

                if ($link = $vendor['link']) {
                    update_term_meta($term_id, 'fdgame_plink', $link);
                }

                if ($year_start = $vendor['year_start']) {
                    update_term_meta($term_id, 'fdgame_pyear_start', $year_start);
                }

                if ($year_end = $vendor['year_end']) {
                    update_term_meta($term_id, 'fdgame_pyear_end', $year_end);
                }

                $images = [];
                if (!empty($vendor['images'])) {
                    $images = ImageService::group_images_by_key($vendor['images'], ImageService::ORIENTATION);
                }
                $img_rectangle = $images['rectangle'][0]['full_url'] ?? null;
                $img_square    = $images['square'][0]['full_url'] ?? null;

                if ($img_rectangle || $img_square) {
                    update_term_meta($term_id, 'fdgame_img', $img_rectangle ?? $img_square);
                }

                $processed++;
            }

            return [
                'status'    => 'success',
                'msg'       => 'Page processed',
                'processed' => $processed,
                'next_page' => $page + 1,
            ];
        } catch (\Exception $exception) {
            return [
                'status' => 'error',
                'msg'    => $exception->getMessage(),
            ];
        } finally {
            if ($didKses) {
                kses_init_filters();
            }
            if ($didDefer) {
                wp_defer_term_counting(false);
                wp_defer_comment_counting(false);
            }
        }
    }
}
