<?php

namespace FreedemoGames\PublicSide;

use FreedemoGames\Core\Render;
use FreedemoGames\Core\ApiClient;
use FreedemoGames\Core\ShortCodes;
use FreedemoGames\Core\GamesService;
use FreedemoGames\Core\ProvidersService;
use FreedemoGames\Core\GamesSource\Entities\GameEntity;
use FreedemoGames\Core\GamesSource\Collection\GamesEntityCollection;
use FreedemoGames\Core\GamesSource\Exceptions\GameDoesntExistException;
use FreedemoGames\Core\GamesSource\Exceptions\InvalidGameArgumentsException;

use function FreedemoGames\fdg_setting;
use function FreedemoGames\fdg_get_partner_token;
use function FreedemoGames\fdg_get_powered_by_logo;

/**
 * Class PublicSideController
 *
 * @package FreedemoGames\PublicSideController
 */
class PublicSideController
{
    const GAMES_PER_PAGE = 40;

    public function __construct()
    {
    }

    /**
     * Register the JavaScript for the public-facing side of the site.
     */
    public function enqueue_scripts()
    {
        $args = [
            'api'              => ApiClient::api_host(),
            'remember_filters' => apply_filters('fdgames/remember_filters', true),
            'lobby_url'        => fdg_setting('lobby-url', site_url()),
            'ajax_url'         => admin_url('admin-ajax.php'),
        ];
        wp_register_script(
            'freedemo-games-slots',
            FREEDEMOGAMES_PLUGIN_URL . 'assets/public/js/freedemo-games-slots.js',
            ['jquery'],
            FREEDEMOGAMES_VERSION,
            true
        );
        wp_localize_script('freedemo-games-slots', 'fdgames_params', $args);
        wp_enqueue_script('freedemo-games-slots');
    }

    public function enqueue_styles()
    {
        wp_enqueue_style(
            'freedemo-games-css',
            FREEDEMOGAMES_PLUGIN_URL . 'assets/public/css/freedemo-games.css',
            [],
            FREEDEMOGAMES_VERSION,
            'all'
        );
    }

    /**
     * @param $atts
     *
     * @return string|null
     */
    public function games_archive($atts): ?string
    {
        $per_page = self::GAMES_PER_PAGE;
        $atts     = shortcode_atts(
            [
                'slots'    => '',
                'type'     => '',
                'order_by' => '',
                'order'    => '',
                'per_page' => $per_page,
            ],
            $atts,
            ShortCodes::SHORT_CODE_GAMES_ARCHIVE
        );

        /** @var GamesEntityCollection $games_collection */
        $games_collection = GamesService::get_games($atts);

        $template = locate_template('templates/freedemo-games/fdgames-archives.view.php');
        if (!$template) {
            $template = FREEDEMOGAMES_PLUGIN_DIR . 'templates/fdgames-archives.view.php';
        }

        $providers_collection = ProvidersService::get_providers();

        return Render::render($template, '', [
            'atts'                 => $atts,
            'games_collection'     => $games_collection,
            'per_page'             => $per_page,
            'providers_collection' => $providers_collection,
        ], true);
    }

    public function init(): void
    {
        add_action('wp_enqueue_scripts', [$this, 'enqueue_styles']);
        add_action('wp_enqueue_scripts', [$this, 'enqueue_scripts']);

        add_shortcode(ShortCodes::SHORT_CODE_GAME_SINGLE, [__CLASS__, 'embed_game']);
        add_shortcode(ShortCodes::SHORT_CODE_GAMES_ARCHIVE, [$this, 'games_archive']);
    }

    /**
     * Single game
     *
     * @param        $atts
     * @param string $content
     *
     * @return string|null
     */
    public static function embed_game($atts, $content = ""): ?string
    {
        try {
            /** @var GameEntity $game */
            $game = GamesService::get_game($atts);
        } catch (InvalidGameArgumentsException|GameDoesntExistException $e) {
            return null;
        }

        $template = locate_template('templates/freedemo-games/fdgames-single-game.view.php');
        if (!$template) {
            $template = FREEDEMOGAMES_PLUGIN_DIR . 'templates/fdgames-single-game.view.php';
        }

        $token    = fdg_get_partner_token();
        $backlink = fdg_setting('fdg-backlink', true);

        return Render::render($template, '', [
            'token'           => $token,
            'game'            => $game,
            'powered_by_logo' => fdg_get_powered_by_logo(),
            'backlink'        => $backlink,
        ], true);
    }
}
